define([
    'underscore',
    'backbone',
    'modules/new-appointment-request/resources/common/model',
    'modules/new-appointment-request/resources/common/facility/model',
    'modules/new-appointment-request/views/common/facility/layout-view',
    'modules/new-appointment-request/views/common/facility/section-view',
    'modules/new-appointment-request/tests/helpers/facility-helper',
    'modules/new-appointment-request/resources/direct/clinics/collection',
    'modules/new-appointment-request/resources/common/pac-team/collection',
], function(_, Backbone, BaseModel, FacilityModel, View, SectionView, helpers, ClinicsCollection, PacTeamCollection) {
    'use strict';


    describe('Scheduling Facility Layout View', function() {

        var TYPE_OF_CARE = 'Zombie Face Lifts';

        // Example: WORCESTER CBOC (WORCESTER, MA)
        // Example: any test name (ANYNAME, XX)
        // The way to display a facility
        var REG_EXP = /^[a-zA-Z ]+ \([a-zA-Z]+, [A-Z]{2}\)$/;


        var view;
        var region;
        var fetch;
        var model;

        var prepareClinicsCollection = function(layoutView) {
            var clinicsCallback = new $.Deferred();
            clinicsCallback.resolve([]);
            layoutView.clinicsCollection = new ClinicsCollection();
            spyOn(layoutView.clinicsCollection, 'fetchAllClinics').and.returnValue(clinicsCallback);
            layoutView.model.set('unfilteredClinics', []);
        };

        var createModel = function() {
            var retval;
            var pacTeamMock = new PacTeamCollection();
            var fDNS   el = new Backbone.Collection();
            fDNS   el.set('facilityCode', 123);
            spyOn(pacTeamMock, 'hasPacTeamAtFacility').and.returnValue(false);
            retval = new BaseModel({
                typeOfCare: new Backbone.Model({
                    id: 'CR1',
                    name: 'whatever',
                }),
                pacTeam: pacTeamMock,
                cityState: new Backbone.Model(),
                fDNS   el: fDNS   el,
            });

            return retval;
        };


        describe('Basic testing and coverage', function() {
            beforeEach(function() {
                fetch = helpers.collectionFetchSpy();
                model = createModel();

                spyOn(View.prototype, 'showRegions').and.callFake(_.noop);
                view = new View({model: model});
                prepareClinicsCollection(view);
                view.collection.set(helpers.mockPatientFacility(), {parse: true});
            });

            it('fetches on creation', function() {
                expect(fetch).toHaveBeenCalled();
            });

            it('triggers show after fetch', function() {
                view.collection.trigger('fetch:success');
                expect(view.showRegions).toHaveBeenCalled();
            });

            it('does not show the section region if the collection is empty', function () {
                var section;

                view.collection.reset();
                view.showBody();
                section = view.getRegions('section');

                expect(section.currentView).toBeUndefined();
            });
        });


        /**
         * https://issues.mobilehealth.DOMAIN.EX/browse/VAR-7864
         *
         * As a user
         * I want to be able to select the specific location where I want to receive care
         * So that my booking process is streamlined focuses on a smaller number of specific clinics
         *
         * When a user views the New Appointment/Request form, the “routing questions” shown at the top part of the form
         * are revised to accommodate location level booking as follows:
         */
        describe('Specification Tests for VAR-7865', function() {
            var gas;

            beforeEach(function() {
                region = helpers.setupTestingRegion();
                fetch = helpers.collectionFetchSpy();
                model = createModel();
                view = new View({
                    model: model,
                    typeOfCareName: TYPE_OF_CARE,
                    googleAnalyticsProvider: {
                        getInstance: function() {
                            gas = jasmine.createSpyObj('gas', ['gas']);
                            return gas;
                        },
                    },
                });
                prepareClinicsCollection(view);
                view.collection.set(helpers.mockPatientFacility(), {parse: true});
                region.show(view);
            });

            afterEach(helpers.resetTestingRegion);

            /**
             * 1. Label: Step 2: Select Location
             */
            describe('1.0.0 - 1.0.0', function() {
                it('shows the correct label', function() {
                    var heading;
                    var legend;
                    var text;

                    view.showRegions();
                    heading = view.getRegion('heading').currentView.el;
                    legend = _.first(heading.getElementsByTagName('legend'));
                    text = legend.innerText;

                    expect(text).toBe('Step 3: Location');
                });
            });


            /**
             * 2.   Instructional text: The following facility locations support scheduling <Type of care selected>
             *      appointments through this app. If you cannot find the location you need, please visit the
             *      [VA Facility Locator] to contact the facility for assistance.
             *
             * 2.2. Hyperlink “VA Facility Locator” to: https://www.DOMAIN.EX/directory/guide/division.asp?dnum=1
             */
            describe('2.0.0 - 2.2.0', function() {
                var text;

                beforeEach(function() {
                    var heading;
                    var instructions;

                    view.showRegions();
                    heading = view.getRegion('heading').currentView.el;
                    instructions = _.first(heading.getElementsByTagName('p'));
                    text = instructions.innerHTML.trim();
                });

                it('creates the basic instructions', function() {
                    expect(text.indexOf('The following facility locations ')).toBe(0);
                });

                it('includes the type of care selected in the instructions', function() {
                    expect(text.indexOf(TYPE_OF_CARE)).toBeGreaterThan(-1);
                });

                it('includes the help url in the instructions', function() {
                    expect(text.indexOf(view.HELP_TEXT)).toBeGreaterThan(-1);
                });
            });


            /**
             * 3.     Label: Select the location you want to visit:
             *
             * 3.1    Drop down list box
             *
             * 3.2    Values: Default drop down list will display all locations (including, children/divisions and the
             *        parent site(s) itself) under the parent site(s) associated with the user’s authentication
             *        credentials where that type of care has been designated as supported in the utility for direct
             *        scheduling, requests, or both methods of scheduling.
             *
             * 3.2.1  Display format should include a “friendly” name configured in the Utility, followed by the city
             *        and state for that location in parenthesis
             *
             * 3.2.2  Example: Boston VAMC (Boston, MA)
             *
             * 3.2.4. Sort the locations alphabetically by State, then City, then Location Name
             *
             */
            describe('3.0.0 - 3.2.4', function() {
                var el;

                beforeEach(function() {
                    view.showRegions();
                    el = view.getRegion('section').currentView.el;
                });

                it('shows the correct label for the dropdown', function() {
                    var label = _.first(el.getElementsByTagName('label'));
                    var text = label.innerText;

                    // The * is because it is a required field.
                    expect(text).toBe('* Select the location you want to visit:');
                });

                it('populates the dropdown', function() {
                    var options = el.getElementsByTagName('option');

                    // There are three items in the collection and a placeholder item that reads 'Select'
                    expect(options.length).toBe(4);
                });

                it('uses the format FRIENDLY_NAME (CITY, STATE) for each dropdown Item', function() {
                    var options = el.getElementsByTagName('option');
                    var last = _.last(options);
                    var text = last.innerText.trim();

                    expect(text).toMatch(REG_EXP);
                });

                // TODO this is going to change to group by city/state then alphabetical
                it('should short the dropdown alphabetically', function() {
                    var _options = el.getElementsByTagName('option');
                    var options = Array.prototype.slice.call(_options);
                    var last;
                    var isPassing;
                    var next;

                    options.shift();

                    last = options.shift();
                    isPassing = true;

                    while (options.length) {
                        next = options.shift();
                        if (next.innerText.localeCompare(last.innerText) < 1) {
                            isPassing = false;
                            break;
                        }
                        last = next;
                    }

                    expect(isPassing).toBeTruthy();
                });
            });


            /**
             * 4.   If only one location is available, select that option by default and display:
             *
             * 4.1. Instructional text: One location supports scheduling <insert type of care> appointments through this
             *      app.
             *
             * 4.2. Display location name (city, state)
             */
            describe('4.0.0 - 4.2.0', function() {
                var spy;
                var dummyReturnView = new Backbone.View();
                var el;

                beforeEach(function() {
                    view.collection.shift();
                    view.collection.shift();
                    spy = spyOn(view, 'getSingleItemView').and.returnValue(dummyReturnView);
                    view.showRegions();
                    el = view.el;
                });

                it('shows the correct instructions for when there is only one matching facility', function() {
                    var p = _.first(el.getElementsByTagName('p'));
                    var text = p.innerText.trim();

                    expect(text.indexOf('One location supports scheduling')).toBe(0);
                    expect(text.indexOf(TYPE_OF_CARE)).toBeGreaterThan(-1);
                    expect(spy).toHaveBeenCalled();
                });

                // it('displays the name of the facility in the format: NAME (CITY, STATE)', function() {
                //     var span = _.last(el.getElementsByTagName('span'));
                //     var text = span.innerText.trim();
                //     expect(text).toMatch(REG_EXP);
                // });
            });


            /**
             * 5.   If no locations are found where the care is supported, display:
             *
             * 5.1. Unfortunately, there are no locations where you are registered that are offering primary care
             *      through this app.
             */
            describe('5.0.0 - 5.1.0', function() {
                it('generates the correct instructions where there are no facilities', function() {
                    var instructions;
                    var frag;
                    var anchors;
                    var anchor;

                    view.collection.reset();
                    instructions = view.getInstructions();
                    frag = document.createElement('div');
                    frag.innerHTML = instructions;
                    anchors = frag.getElementsByTagName('a');
                    anchor = _.first(anchors);

                    // Shows an error message and displays a link to a help page.
                    expect(instructions).toContain('Unfortunately');
                    expect(anchor.href).toEqual('https://www.DOMAIN.EX/directory/guide/division.asp?dnum=1');
                });
            });
        });

        /**
         * https://issues.mobilehealth.DOMAIN.EX/browse/VAR-8702
         *
         * Given a user has selected Express Care from the Type of Care drop down, when the user
         * selects a location (or the default location is displayed), display the following text
         * below the location selection question: Express Care request at this location must be
         * submitted before <end time> <time zone>.
         */
        describe('Specification Tests for VAR-8702', function() {
            var time;
            var facilityModel;
            var expectedText;

            beforeEach(function() {
                time = '13:00 EST';
                facilityModel = new FacilityModel({
                    expressTimes: {
                        start: '08:00',
                        end: '13:00',
                        timezone: 'EST',
                        offsetUtc: '-05:00',
                    },
                }, {parse: true});
                expectedText = 'Express Care request at this location must be submitted before ' + time + '.';
            });

            describe('facility layout view', function() {
                it('generates the correct express message for a single facility', function() {
                    var _view = new View({
                        model: new BaseModel({
                            typeOfCare: new Backbone.Model({
                                id: 'CR1',
                                name: 'whatever',
                            }),
                            cityState: new Backbone.Model(),
                        }),
                        typeOfCareName: TYPE_OF_CARE,
                    });
                    var output = _view.getExpressMessage('CR1', facilityModel);
                    prepareClinicsCollection(view);

                    expect(output).toBe(expectedText);
                });
            });

            describe('facility section view', function() {
                var sectionView;
                var gas;

                beforeEach(function() {
                    var sectionRegion;

                    region = helpers.setupTestingRegion();
                    fetch = helpers.collectionFetchSpy();
                    model = createModel();
                    view = new View({
                        model: model,
                        typeOfCareName: TYPE_OF_CARE,
                        googleAnalyticsProvider: {
                            getInstance: function() {
                                gas = jasmine.createSpyObj('gas', ['gas']);
                                return gas;
                            },
                        },
                    });
                    prepareClinicsCollection(view);
                    view.collection.set(helpers.mockPatientFacility(), {parse: true});
                    region.show(view);
                    view.showRegions();

                    sectionRegion = view.getRegion('section');
                    sectionView = sectionRegion.currentView;
                });

                afterEach(helpers.resetTestingRegion);

                it('appends the dom with the express message after facility is selected', function() {
                    var output;

                    sectionView._insertExpressCareMessage(facilityModel);
                    output = sectionView.ui.express.text();

                    expect(output).toBe(expectedText);
                });
            });
        });

        describe('Section View Test', function() {
            var sectionView;
            var gas;

            beforeEach(function() {
                var sectionRegion;

                region = helpers.setupTestingRegion();
                fetch = helpers.collectionFetchSpy();
                model = createModel();

                view = new View({
                    model: model,
                    typeOfCareName: TYPE_OF_CARE,
                    googleAnalyticsProvider: {
                        getInstance: function() {
                            gas = jasmine.createSpyObj('gas', ['gas']);
                            return gas;
                        },
                    },
                });
                prepareClinicsCollection(view);
                view.collection.set(helpers.mockPatientFacility(), {parse: true});
                region.show(view);
                view.showRegions();

                sectionRegion = view.getRegion('section');
                sectionView = sectionRegion.currentView;
            });

            afterEach(helpers.resetTestingRegion);

            it('updates its model on a collection change', function() {
                var isCalled = false;
                var id = 'myValue';
                var value = view.collection.first().get('institutionCode');

                view.listenTo(view.model, 'change:myValue', function() {
                    isCalled = true;
                });

                sectionView.collection.trigger('change:value', new Backbone.Model({
                    id: id,
                    value: value,
                }));

                expect(isCalled).toBeTruthy();
                expect(gas.gas).toHaveBeenCalled();
            });

            it('should call Google Analytics when the collection changes upon selecting a facility', function() {
                var id = 'id';
                var value = view.collection.first().get('institutionCode');

                sectionView.collection.trigger('change:value', new Backbone.Model({
                    id: id,
                    value: value,
                }));

                expect(gas.gas).toHaveBeenCalledWith('send', 'event', 'veteran-appointment', 'facility-selected', 'GREENFIELD (CBOC) (GREENFIELD, MA)');
            });
        });
    });

    describe('Scheduling Facility Section View', function() {
        var facilityQuestionModelSpy;

        var setup = function(sectionDataIn) {
            var facilitiesCollectionMock;
            var facilityModelMock;
            var view;
            var googleAnalyticsProviderMock;
            facilityQuestionModelSpy = jasmine.createSpyObj('facilityQuestionModelSpy', ['set']);

            facilitiesCollectionMock = {
                toSectionData: function () {
                    return sectionDataIn;
                },
            };
            facilityModelMock = {
                pacTeam: function() {
                    return {};
                },
            };
            googleAnalyticsProviderMock = {
                getInstance: function() {
                    return jasmine.createSpy('googleAnalyticsMock', ['gas']);
                },
            };

            view = new SectionView({
                model: facilityModelMock,
                patientEnrolledFacilities: facilitiesCollectionMock,
                clinicsInstitutionCodes: [],
                googleAnalyticsProvider: googleAnalyticsProviderMock,
            });

            view.collection = {
                get: function(property) {
                    if (property === 'facility') {
                        return facilityQuestionModelSpy;
                    }
                    return null;
                },
            };

            return view;
        };

        it('should put the facilities with PAC Teams first in the dropdown', function() {
            var facility1 = { 'text': 'facility1', 'hasPactLabel': false };
            var facility2 = { 'text': 'facility2', 'hasPactLabel': false };
            var facility3 = { 'text': 'facility3', 'hasPactLabel': false };
            var facility4 = { 'text': 'facility4', 'hasPactLabel': true };
            var facility5 = { 'text': 'facility5', 'hasPactLabel': true };

            var expectedOrder = [facility4, facility5, facility1, facility2, facility3];
            var sectionDataIn = [facility1, facility2, facility3, facility4, facility5];
            var view = setup(sectionDataIn);

            view.setFacilityQuestionModel();

            expect(facilityQuestionModelSpy.set).toHaveBeenCalledWith('values', expectedOrder);
        });

        it('should sort the facilities alphabetically', function() {
            var facilityA = { 'text': 'a', 'hasPactLabel': false };
            var facilityB = { 'text': 'b', 'hasPactLabel': false };
            var facilityC = { 'text': 'c', 'hasPactLabel': false };

            var expectedOrder = [facilityA, facilityB, facilityC];
            var sectionDataIn = [facilityA, facilityC, facilityB];
            var view = setup(sectionDataIn);

            view.setFacilityQuestionModel();

            expect(facilityQuestionModelSpy.set).toHaveBeenCalledWith('values', expectedOrder);
        });

    });
});
TTTT